#!/bin/ruby

## Description
#		Incorporates changes to a previously signed file in such a way that the 
#		changes can be unwrapped later on in order to revert to a previously signed
#		version. The digital signature, publisher's certificate and diff output are
#		added to file outputFile.ds


require 'optparse'
require 'ostruct'

class OptparseClass
	#
	#	Return structure describing the options.
	#
	def self.parse(args)
		# Set default options
		options = OpenStruct.new
		options.n = false
		options.verbose = false
		options.orgFile = ""
		options.modFile = ""
		options.outFile = ""
		options.myCertificate = ""
		options.myPrivateKey = ""

		opts = OptionParser.new do |opts|
			opts.banner = "Usage csignwrap.rb [options] myCertificate myPrivateKey orgfile modfile"

			opts.separator ""
			opts.separator "Specific options:"
		
			# Boolean switches
			opts.on("-v", "--[no-]verbose", "Run verbosely") do |v|
				options.verbose = v
			end

			opts.separator ""
			opts.separator "Common options:"
		
			opts.on_tail("-h", "--help", "Show this message") do
				puts opts
				exit
			end
		end
		
		opts.parse!(args)
		# Handle file arguments
		if ARGV.count != 4 then
			puts opts
			exit
		else
			options.myCertificate = ARGV[0]
			options.myPrivateKey = ARGV[1]
			options.orgFile = ARGV[2]
			options.modFile = ARGV[3]
		end
		ARGV.each do |file|
			if not File.file? file then
				puts "Cannot open file: #{file}"
				exit
			end
		end

		options.outFile = "#{options.modFile}.ds" if options.outFile = ""
		options
	end	# parse()
end	# class OptparseClass

options = OptparseClass.parse(ARGV)
if File.file? options.outFile and not options.new then
	puts "Signature operation could not be completed since #{options.outFile} does not exist."
	puts "Must have previous signature to perform wrapping."
	exit
end

#
#	The actual work
#
print "Calculating differences..." if options.verbose
diff = `diff #{options.orgFile} #{options.modFile}`
puts "done." if options.verbose

print "Computing and signing hash..." if options.verbose
signature = `openssl dgst -md5 < #{options.modFile} | openssl rsautl -sign -inkey #{options.myPrivateKey} | openssl enc -base64`
puts "done." if options.verbose

print "Writing to file..." if options.verbose
open(options.outFile, 'w') do |f|
	f << "-----BEGIN DIFF-----\n" << diff << "-----END DIFF-----\n"
	f << "-----BEGIN SIGNATURE-----\n" << signature << "-----END SIGNATURE-----\n"
	open(options.myCertificate).each {|x| f << x}
end

print "done\n" if options.verbose

