#!/bin/ruby

## Description
#		Generate a digital signature (encrypted hash) of the source code file using
#		public certificate myCertificate and private key myPrivateKey

require 'optparse'
require 'ostruct'

class OptparseClass
	#
	#	Return structure describing the options.
	#
	def self.parse(args)
		# Set default options
		options = OpenStruct.new
		options.n = false
		options.verbose = false
		options.filename = ""
		options.outFile = ""
		options.myCertificate = ""
		options.myPrivateKey = ""

		opts = OptionParser.new do |opts|
			opts.banner = "Usage csigncode.rb [options] myCertificate myPrivateKey filename"

			opts.separator ""
			opts.separator "Specific options:"
		
			# Options with mandatory arguments
			opts.on("-o", "--out FILENAME", 
							"Write to FILENAME instead of default") do |file|
								options.outFile = file
			end

			# Boolean switches
			opts.on("-v", "--[no-]verbose", "Run verbosely") do |v|
				options.verbose = v
			end

			opts.on("-n", "--new", "Overwrite existing signature") do |n|
				options.new = n
			end

			opts.separator ""
			opts.separator "Common options:"
		
			opts.on_tail("-h", "--help", "Show this message") do
				puts opts
				exit
			end
		end
		
		opts.parse!(args)
		# Handle file arguments
		if ARGV.count != 3 then
			puts opts
			exit
		else
			options.myCertificate = ARGV[0]
			options.myPrivateKey = ARGV[1]
			options.filename = ARGV[2]
		end
		ARGV.each do |file|
			if not File.file? file then
				puts "Cannot open file: #{file}"
				exit
			end
		end

		options.outFile = "#{options.filename}.ds" if options.outFile = ""
		options
	end	# parse()
end	# class OptparseClass

options = OptparseClass.parse(ARGV)
if File.file? options.outFile and not options.new then
	puts "Signature operation could not be completed since #{options.outFile} already exits.
Use -n option to overwrite the current signature with a new signature."
	exit
end

#
#	The actual work
#
print "Computing and signing hash..." if options.verbose
signature = `openssl dgst -md5 < #{options.filename} | openssl rsautl -sign -inkey #{options.myPrivateKey} | openssl enc -base64`
print "done\nWriting to file..." if options.verbose
open(options.outFile, 'w') do |f|
	f << "-----BEGIN SIGNATURE-----\n"
	f << signature
	f << "-----END SIGNATURE-----\n"
	open(options.myCertificate).each {|x| f << x}
end

print "done\n" if options.verbose

